# @summary installs matlab
#
# Does the heavy lifting to install matlab on target machine
#
# @example
#   include matlab::install


class matlab::install(      # $namevar matlab release version
  $iso = '/systems/images/R2019a_Linux.iso',        # matlab iso for installation
  $licensekey = '46435-40118-41073-52403-09836-46767-19152-32823-45696-25667-33707-32533-26787-25036-50875-55401-43672-61214-02211-03677-62763-55854',   # format: #####-#####-#####-#####
  $licensefile = 'puppet:///modules/matlab/license.lic',      # license.lic as provided by mathworks
  $licenseagree = true,    # do you agree to license, true/false ?
  $prefix = '/opt'    # install prefix
) {
  include 'matlab'
  #include matlab::vardir
  #$vardir = $::matlab::vardir::module_vardir	# with trailing slash
  $vardir = "/opt/matlab_install"

  # individual install name
  $name_ = 'R2019a'

  $install_destination = "${prefix}/MATLAB/${name_}"

  # does user accept license ?
  $agree = $licenseagree ? {
    true => 'yes',
    default => 'no',
  }
  
  # make install folder
  file { "/opt/matlab_install":
    ensure  => directory,    # make sure this is a directory
    recurse => false,    # don't manage directory
    purge   => false,      # don't purge unmanaged files
    force   => false,      # don't purge subdirs and links
    owner   => root,
    group   => root,
    alias   => "install_folder",
  }

  # make folder to mount on
  file { "/mnt/matlab-${name_}":
    ensure  => directory,    # make sure this is a directory
    recurse => false,    # don't manage directory
    purge   => false,      # don't purge unmanaged files
    force   => false,      # don't purge subdirs and links
    owner   => root,
    group   => root,
    mode    => '0555',      # default for iso mounts
    backup  => false,    # don't backup to filebucket
    require => File["install_folder"],
  }

  # get iso to mountUpdate Puppetfile
  # TODO: since there seem to be different iso's for each version, maybe
  # we should add a unique identifier based on the $iso variable here.
  file { "${vardir}/MATHWORKS-${name_}.iso":
    ensure  => present,
    owner   => root,
    group   => root,
    mode    => '0600',    # u=rw,go=
    backup  => false,  # don't backup to filebucket!
    source  => $iso,
    alias   => "matlab_iso.${name_}",
    require => File["/mnt/matlab-${name_}"],
  }

  # mount!
  # TODO: replace this mount with an exec that has an:
  # onlyif => the_binary_is_not_installed so that a normal machine
  # doesn't need to have the iso mounted all the time...
  mount { "/mnt/matlab-${name_}":
    ensure  => mounted,
    atboot  => true,
    device  => "${vardir}/MATHWORKS-${name_}.iso",
    fstype  => 'iso9660',
    options => 'loop,ro',
    dump    => '0',    # fs_freq: 0 to skip file system dumps
    pass    => '0',    # fs_passno: 0 to skip fsck on boot
    alias   => "matlab_mount.${name_}",
    require => [File["matlab_iso.${name_}"]],
  }

  # build installer parameters file in our scratch directory
  file { "${vardir}/installer_input.txt.${name_}":
    ensure  => present,
    owner   => root,
    group   => root,
    mode    => '0600',  # u=rw,go=r
    require => Mount["matlab_mount.${name_}"],
    source => 'puppet:///modules/matlab/installer_input.txt',
    alias   => "matlab_input.${name_}",
  }

  # install matlab
  exec { "/mnt/matlab-${name_}/install -inputFile ${vardir}/installer_input.txt.${name_}":
    logoutput => on_failure,
    creates   => $install_destination,  # when this folder appears, we assume it got installed
    require   => File["matlab_input.${name_}"],
    alias     => "matlab_install.${name_}",
    timeout   => 3600,
  }
  
  # create activation .ini
  file { "${install_destination}/bin/activate.ini":
    ensure  => present,
    owner   => root,
    group   => root,
    mode    => '0644',
    source  => 'puppet:///modules/matlab/activate.ini',
    require => Exec["matlab_install.${name_}"],
  }

  # create 'licenses' directory
  file { "${install_destination}/licenses/":
    ensure  => directory,    # make sure this is a directory
    recurse => true,    # recursively manage directory
    purge   => true,      # purge all unmanaged files
    force   => true,      # also purge subdirs and links
    owner   => root,
    group   => root,
    mode    => '0644',
    backup  => false,    # don't backup to filebucket
    require => Exec["matlab_install.${name_}"],
  }

  # copy over license file to activate
  #file { "${install_destination}/licenses/license.lic":
  #  ensure  => present,
  #  owner   => root,
  #  group   => root,
  #  # TODO: is there a worry that someone will steal the license ?
  #  mode    => '0644',    # u=rw,g=r,o=r
  #  source  => $licensefile,
  #  require => File["${install_destination}/licenses/"],
  #}
}