# @summary installs matlab
#
# Does the heavy lifting to install matlab on target machine
#
# @example
#   include matlab::install


define matlab::install(      # $namevar matlab release version
  $iso = 'R2019a_Linux.iso',        # matlab iso for installation
  $licensekey = '46435-40118-41073-52403-09836-46767-19152-32823-45696-25667-33707-32533-26787-25036-50875-55401-43672-61214-02211-03677-62763-55854',   # format: #####-#####-#####-#####
  $licensefile = 'puppet:///modules/matlab/license.lic',      # license.lic as provided by mathworks
  $licenseagree = true,    # do you agree to license, true/false ?
  $prefix = '/opt'    # install prefix
) {
  include 'matlab'
  include matlab::vardir
  #$vardir = $::matlab::vardir::module_vardir	# with trailing slash
  $vardir = regsubst($::matlab::vardir::module_vardir, '\/$', '')

  $install_destination = "${prefix}/MATLAB/${name}"

  # does user accept license ?
  $agree = $licenseagree ? {
    true => 'yes',
    default => 'no',
  }

  # make folder to mount on
  file { "/mnt/matlab-${name}":
    ensure  => directory,    # make sure this is a directory
    recurse => false,    # don't manage directory
    purge   => false,      # don't purge unmanaged files
    force   => false,      # don't purge subdirs and links
    owner   => root,
    group   => root,
    mode    => '0555',      # default for iso mounts
    backup  => false,    # don't backup to filebucket
  }

  # get iso to mount
  # TODO: since there seem to be different iso's for each version, maybe
  # we should add a unique identifier based on the $iso variable here.
  file { "${vardir}/MATHWORKS-${name}.iso":
    ensure  => present,
    owner   => root,
    group   => nobody,
    mode    => '0600',    # u=rw,go=
    backup  => false,  # don't backup to filebucket!
    source  => $iso,
    alias   => "matlab_iso.${name}",
    require => File["/mnt/matlab-${name}"],
  }

  # mount!
  # TODO: replace this mount with an exec that has an:
  # onlyif => the_binary_is_not_installed so that a normal machine
  # doesn't need to have the iso mounted all the time...
  mount { "/mnt/matlab-${name}":
    ensure  => mounted,
    atboot  => true,
    device  => "${vardir}/MATHWORKS-${name}.iso",
    fstype  => 'iso9660',
    options => 'loop,ro',
    dump    => '0',    # fs_freq: 0 to skip file system dumps
    pass    => '0',    # fs_passno: 0 to skip fsck on boot
    alias   => "matlab_mount.${name}",
    require => [File["matlab_iso.${name}"]],
  }

  # build installer parameters file in our scratch directory
  file { "${vardir}/installer_input.txt.${name}":
    ensure  => present,
    owner   => root,
    group   => nobody,
    mode    => '0600',  # u=rw,go=r
    require => Mount["matlab_mount.${name}"],
    content => template('matlab/installer_input.txt.erb'),
    alias   => "matlab_input.${name}",
  }

  # install matlab
  exec { "/mnt/matlab-${name}/install -inputFile ${vardir}/installer_input.txt.${name}":
    logoutput => on_failure,
    creates   => $install_destination,  # when this folder appears, we assume it got installed
    require   => File["matlab_input.${name}"],
    alias     => "matlab_install.${name}",
  }

  # create 'licenses' directory
  file { "${install_destination}/licenses/":
    ensure  => directory,    # make sure this is a directory
    recurse => true,    # recursively manage directory
    purge   => true,      # purge all unmanaged files
    force   => true,      # also purge subdirs and links
    owner   => root,
    group   => root,
    mode    => '0644',
    backup  => false,    # don't backup to filebucket
    require => Exec["matlab_install.${name}"],
  }

  # copy over license file to activate
  file { "${install_destination}/licenses/license.lic":
    ensure  => present,
    owner   => root,
    group   => nobody,
    # TODO: is there a worry that someone will steal the license ?
    mode    => '0644',    # u=rw,g=r,o=r
    source  => $licensefile,
    require => File["${install_destination}/licenses/"],
  }
}